function [mu,dmudtheta] = poissonrate(rateparams,theta)
% Returns the Poisson-rates for all pixels and all first order
% derivatives w.r.t. the parameters theta.

% (C) Copyright 2018
% All rights reserved
% Department of Imaging Physics
% Faculty of Applied Sciences
% Delft University of Technology
% Delft, The Netherlands   

% parameter settingsc
Nxpixels = rateparams.Nxpixels;
Nypixels = rateparams.Nypixels;
pixelsize = rateparams.pixelsize;
ImageSizex = Nxpixels*pixelsize/2;
ImageSizey = Nypixels*pixelsize/2;
ximag = -ImageSizex+pixelsize/2:pixelsize:ImageSizex;
yimag = -ImageSizey+pixelsize/2:pixelsize:ImageSizey;
[X,Y] = meshgrid(ximag,yimag);

% variables and (dummy) parameters depending on PSF model
model = rateparams.model;
switch model
  case 'standard2D'
    minorder = 0;
    maxorder = 0;
    convpostolam = 1;
    lambda = 500;
    x0 = theta(1);
    y0 = theta(2);
    sigma = theta(3);
    sigma_x = theta(3);
    sigma_y = theta(3);
    Nph = theta(4);
    bg = theta(5);
  case 'astigmatic3D'
    minorder = 0;
    maxorder = 0;
    convpostolam = 1;
    lambda = 500;
    x0 = theta(1);
    y0 = theta(2);
    sigma = theta(3);
    Nph = theta(4);
    bg = theta(5);
    F = theta(6);
    sigmaxrat = sqrt(1+F);
    sigmayrat = sqrt(1-F);
    sigma_x = sigma*sigmaxrat;
    sigma_y = sigma*sigmayrat;
  case 'diffractivecolor2D'
    minorder = -1;
    maxorder = 1;
    convpostolam = rateparams.convpostolam;
    phasedepth = rateparams.phasedepth;
    x0 = theta(1);
    y0 = theta(2);
    sigma = theta(3);
    sigma_x = theta(3);
    sigma_y = theta(3);
    Nph = theta(4);
    bg = theta(5);
    lambda = theta(6);
    sinf = sin(pi*phasedepth/lambda);
    cosf = cos(pi*phasedepth/lambda);
    diffeff = [(4/pi^2)*sinf^2, cosf^2, (4/pi^2)*sinf^2];
    ddiffeffdlam = [-4/pi^2, 1, -4/pi^2]*(2*pi*phasedepth*sinf*cosf/lambda^2);
  case 'diffractivecolor3D'
    minorder = -1;
    maxorder = 1;
    convpostolam = rateparams.convpostolam;
    phasedepth = rateparams.phasedepth;
    Fmax = rateparams.Fmax;
    x0 = theta(1);
    y0 = theta(2);
    sigma = theta(3);
    Nph = theta(4);
    bg = theta(5);
    lambda = theta(6);
    F = theta(7);
    sinf = sin(pi*phasedepth/lambda);
    cosf = cos(pi*phasedepth/lambda);
    diffeff = [(4/pi^2)*sinf^2, cosf^2, (4/pi^2)*sinf^2];
    ddiffeffdlam = [-4/pi^2, 1, -4/pi^2]*(2*pi*phasedepth*sinf*cosf/lambda^2);
    dallsigsdsig = ones(maxorder-minorder+1,1);
    dallsigsdF = ones(maxorder-minorder+1,1);
    zf = Fmax+sqrt(Fmax^2-F^2);
    dallsigsdsig(1) = sqrt((1+F)/zf); 
    dallsigsdsig(2) = sqrt(1/zf-Fmax/2);
    dallsigsdsig(3) = sqrt((1-F)/zf);
    allsigs = sigma*dallsigsdsig;
    dallsigsdF(1) = 1/zf+(1+F)*F/zf/(zf-Fmax);
    dallsigsdF(2) = F/zf/(zf-Fmax);
    dallsigsdF(3) = -1/zf+(1-F)*F/zf/(zf-Fmax);
    dallsigsdF = dallsigsdF./dallsigsdsig/2;
end

% loop over the diffraction orders
numparams = length(theta);
mu = bg*ones(Nypixels,Nxpixels);
dmudtheta = zeros(Nypixels,Nxpixels,numparams);

for jorder = minorder:maxorder

% diffraction order dependent spot width    
  switch model
    case 'diffractivecolor3D'
      sigma_x = allsigs(jorder-minorder+1);
      sigma_y = allsigs(jorder-minorder+1);
  end
    
% Evaluation of pixel integrals of Hermite functions.
  upx = (X+pixelsize/2-x0-jorder*lambda/convpostolam)/(sqrt(2)*sigma_x);
  umx = (X-pixelsize/2-x0-jorder*lambda/convpostolam)/(sqrt(2)*sigma_x);
  upy = (Y+pixelsize/2-y0)/(sqrt(2)*sigma_y);
  umy = (Y-pixelsize/2-y0)/(sqrt(2)*sigma_y);

% initialization.
  delpsix = cell(2,1);
  delupsix = cell(2,1);
  delpsiy = cell(2,1);
  delupsiy = cell(2,1);
  delpsi2d = cell(2,2);
  delupsi2d = cell(2,2);

  for l = 0:1
     delpsix{l+1} = mypsi(upx,l-1)-mypsi(umx,l-1);
     delupsix{l+1} = upx.*mypsi(upx,l-1)-umx.*mypsi(umx,l-1); 
  end

  for m = 0:1
     delpsiy{m+1} = mypsi(upy,m-1)-mypsi(umy,m-1); 
     delupsiy{m+1} = upy.*mypsi(upy,m-1)-umy.*mypsi(umy,m-1); 
  end

  for l = 0:1
    for m = 0:1
      delpsi2d{l+1,m+1} = delpsix{l+1}.*delpsiy{m+1}; 
    end
  end

  for l = 0:0
    for m = 0:0
      delupsi2d{l+1,m+1} = delupsix{l+2}.*delpsiy{m+1}+delpsix{l+1}.*delupsiy{m+2}; 
    end
  end

% Evaluation of Poisson-rate and first order derivatives

  switch model
    case 'standard2D'
      mu = Nph*delpsi2d{1,1}/pi+bg;   
      dmudtheta(:,:,1) = Nph*delpsi2d{2,1}/(sqrt(2)*pi*sigma);
      dmudtheta(:,:,2) = Nph*delpsi2d{1,2}/(sqrt(2)*pi*sigma);
      dmudtheta(:,:,3) = Nph*delupsi2d{1,1}/(pi*sigma);
      dmudtheta(:,:,4) = delpsi2d{1,1}/pi;
      dmudtheta(:,:,5) = 1;
    case 'astigmatic3D'
      mu = Nph*delpsi2d{1,1}/pi+bg;
      dmudtheta(:,:,1) = Nph*delpsi2d{2,1}/(sqrt(2)*pi*sigma_x);
      dmudtheta(:,:,2) = Nph*delpsi2d{1,2}/(sqrt(2)*pi*sigma_y);
      dmudsigmax = Nph/(pi*sigma_x)*delpsiy{1}.*delupsix{2};
      dmudsigmay = Nph/(pi*sigma_y)*delpsix{1}.*delupsiy{2};
      dmudtheta(:,:,3) = sigmaxrat*dmudsigmax+sigmayrat*dmudsigmay;
      dmudtheta(:,:,4) = delpsi2d{1,1}/pi;
      dmudtheta(:,:,5) = 1;
      dmudtheta(:,:,6) = (sigma_x/sigmaxrat/2)*dmudsigmax-(sigma_y/sigmayrat/2)*dmudsigmay;  
    case 'diffractivecolor2D'
      mu = mu+Nph*diffeff(jorder-minorder+1)*delpsi2d{1,1}/pi;
      dmudtheta(:,:,1) = dmudtheta(:,:,1)+Nph*diffeff(jorder-minorder+1)*delpsi2d{2,1}/(sqrt(2)*pi*sigma);
      dmudtheta(:,:,2) = dmudtheta(:,:,2)+Nph*diffeff(jorder-minorder+1)*delpsi2d{1,2}/(sqrt(2)*pi*sigma);
      dmudtheta(:,:,3) = dmudtheta(:,:,3)+Nph*diffeff(jorder-minorder+1)*delupsi2d{1,1}/(pi*sigma);
      dmudtheta(:,:,4) = dmudtheta(:,:,4)+diffeff(jorder-minorder+1)*delpsi2d{1,1}/pi;
      dmudtheta(:,:,5) = 1;
      dmudtheta(:,:,6) = dmudtheta(:,:,6)+...
          jorder*Nph*diffeff(jorder-minorder+1)*delpsi2d{2,1}/(sqrt(2)*pi*sigma)/convpostolam+...
          Nph*ddiffeffdlam(jorder-minorder+1)*delpsi2d{1,1}/pi;
    case 'diffractivecolor3D'
      mu = mu+Nph*diffeff(jorder-minorder+1)*delpsi2d{1,1}/pi;
      dmudtheta(:,:,1) = dmudtheta(:,:,1)+Nph*diffeff(jorder-minorder+1)*delpsi2d{2,1}/(sqrt(2)*pi*sigma);
      dmudtheta(:,:,2) = dmudtheta(:,:,2)+Nph*diffeff(jorder-minorder+1)*delpsi2d{1,2}/(sqrt(2)*pi*sigma);
      dmudtheta(:,:,3) = dmudtheta(:,:,3)+...
          Nph*dallsigsdsig(jorder-minorder+1)*diffeff(jorder-minorder+1)*delupsi2d{1,1}/(pi*sigma);
      dmudtheta(:,:,4) = dmudtheta(:,:,4)+diffeff(jorder-minorder+1)*delpsi2d{1,1}/pi;
      dmudtheta(:,:,5) = 1;
      dmudtheta(:,:,6) = dmudtheta(:,:,6)+...
          jorder*Nph*diffeff(jorder-minorder+1)*delpsi2d{2,1}/(sqrt(2)*pi*sigma)/convpostolam+...
          Nph*ddiffeffdlam(jorder-minorder+1)*delpsi2d{1,1}/pi;
      dmudtheta(:,:,7) = dmudtheta(:,:,7)+...
          Nph*dallsigsdF(jorder-minorder+1)*diffeff(jorder-minorder+1)*delupsi2d{1,1}/(pi*sigma);
  end

end